/* global sc, Packages */

function moduleFunction(config) {
  // Initialize $$ with provided config or default
  let $$ = config || require('./debug')({ debug: 'cache', path: module.id });

  let metadata = require('common/metadata');
  let objects = require('common/objects').sc($$.environment.context || sc);

  // Java imports
  let PDFMergerUtility = global.Packages.org.apache.pdfbox.util.PDFMergerUtility;
  let PDFTextStripper = global.Packages.org.apache.pdfbox.util.PDFTextStripper;
  let PDDocument = global.Packages.org.apache.pdfbox.pdmodel.PDDocument;
  let PdfLoader = global.Packages.agorum.roi.pdf.PdfLoader;
  let PDPage = global.Packages.org.apache.pdfbox.pdmodel.PDPage;
  let PDDocumentOutline = Packages.org.apache.pdfbox.pdmodel.interactive.documentnavigation.outline.PDDocumentOutline;
  let PDOutlineItem = Packages.org.apache.pdfbox.pdmodel.interactive.documentnavigation.outline.PDOutlineItem;

  /**
   * Validates a PDF document.
   *
   * @param {object} object - The object to validate.
   * @param {pdDocument} pdDocument - The PDF document to validate.
   * @throws {Error} If the PDF document is encrypted or has no pages.
   * @returns {boolean} Returns true if the validation passes.
   */
  function defaultValidation(object, pdDocument) {
    if (pdDocument.isEncrypted()) {
      throw new Error('pdf document is encrypted');
    }
    if (pdDocument.getNumberOfPages() === 0) {
      throw new Error('pdf document has no pages');
    }
    return true;
  }

  /**
   * Loads a PDF document from an object.
   *
   * @param {string} _object - The object to load the PDF from.
   * @param {Object} _opts - Optional parameters for loading the PDF.
   * @returns {Object} - The loaded PDF document.
   */
  function loadFromObject(_object, _opts) {
    let opts = Object.assign(
      {
        error: null,
        success: null,
        validate: defaultValidation,
      },
      _opts || {}
    );
    let pdDocument = null;
    let error = null;
    let object = objects.tryFind(_object);
    if (!object) {
      error = new Error('pdf object not found');
      if (opts.error) opts.error(object, error);
    } else {
      try {
        pdDocument = PdfLoader.load(object.contentStream);
        if (opts.validate) {
          try {
            opts.validate(object, pdDocument);
          } catch (e) {
            error = e;
            if (opts.error) {
              opts.error(object, error);
            }
          }
        }
        if (opts.success) {
          opts.success(object, pdDocument);
        }
      } catch (e) {
        error = e;
        if (opts.error) {
          opts.error(object, error);
        }
      }
    }
    return pdDocument;
  }

  /**
   * Removes bookmarks from a PDF document.
   * @param {Object} pdDocument - The PDF document object.
   */
  function removeBookmarks(pdDocument) {
    if (pdDocument && pdDocument.getDocumentCatalog()) {
      let outline = pdDocument.getDocumentCatalog().getDocumentOutline();
      if (outline != null) {
        outline = new PDDocumentOutline();
        pdDocument.getDocumentCatalog().setDocumentOutline(outline);
      }
    }
  }

  /**
   * Merges multiple PDF documents based on a query expression.
   *
   * @param {Object} _object - The target object where the merged PDF will be saved.
   * @param {Object} _opts - Optional parameters for the merge operation.
   * @returns {Object} The target object where the merged PDF is saved.
   */
  function mergeDocumentsFromQuery(_object, _opts) {
    let opts = Object.assign(
      {
        success: null,
        validate: null,
        error: null,
        sort: null,
        query: null,
      },
      _opts || {}
    );

    if (!opts.query) {
      throw new Error('pdf.mergeDocuments - query: missing query expression');
    }

    let object = objects.tryFind(_object);
    if (!object) {
      throw new Error('pdf.mergeDocuments - find: pdf target object not found');
    }

    let pdfMergerUtility = new PDFMergerUtility();
    let targetDocument = new PDDocument();
    let catalog = targetDocument.getDocumentCatalog();
    let outline = new PDDocumentOutline();
    catalog.setDocumentOutline(outline);

    let lastObject = {};
    let bookmarks = [
      {
        key: null,
        bookmark: outline,
        child: null,
      },
    ];

    bookmarks = bookmarks.concat(
      opts.sort.map(function(level) {
        return {
          key: level,
          bookmark: null,
          child: null,
        };
      })
    );

    function addBookmark(page, obj) {
      let currentObject = metadata().load(obj).data();
      bookmarks.forEach(function(bookmark, index) {
        if (index > 0) {
          if (currentObject[bookmark.key] !== lastObject[bookmark.key]) {
            bookmarks[index].bookmark = new PDOutlineItem();
            bookmarks[index].bookmark.setTitle(currentObject[bookmark.key]);
            bookmarks[index].bookmark.setDestination(page);
            if (bookmarks[index - 1].bookmark) {
              bookmarks[index - 1].bookmark.appendChild(bookmarks[index].bookmark);
            }
            bookmarks[index].title = currentObject[bookmark.key];
            lastObject = {};
          }
        }
      });
      lastObject = currentObject;
    }

    let searchQuery = objects.query(opts.query);
    if (opts.sort) {
      searchQuery._sort = opts.sort.map(function(key) {
        return key.toLowerCase();
      });
    }

    searchQuery.forEach(function(obj) {
      let pdDocument = loadFromObject(obj, opts);
      if (pdDocument) {
        try {
          removeBookmarks(pdDocument);
          pdfMergerUtility.appendDocument(targetDocument, pdDocument);
          try {
            let pdPages = pdDocument.getNumberOfPages();
            let targetPages = targetDocument.getNumberOfPages();
            let bookmarkPage = catalog.getAllPages().get(targetPages - pdPages);
            addBookmark(bookmarkPage, obj);
          } catch (e) {
            if (opts.error) {
              opts.error(object, new Error('bookmark - ' + e.message));
            }
          }
        } catch (e) {
          if (opts.error) {
            opts.error(object, new Error('merge - ' + e.message));
          }
        } finally {
          pdDocument.close();
        }
      }
    });

    try {
      outline.openNode();
      targetDocument.save(object.contentOutputStream);
      targetDocument.close();
    } catch (e) {
      if (opts.error) {
        opts.error(object, new Error('save - ' + e.message));
      }
    }
    return object;
  }

  /**
   * Extracts text from a PDF document.
   *
   * @param {string} _obj - The object to extract text from.
   * @param {Object} _opts - The options for text extraction.
   * @returns {string} The extracted text from the PDF document.
   */
  function extractText(_obj, _opts) {
    let obj = objects.find(_obj);
    let opts = Object.assign(
      {
        fromPage: null,
        toPage: null,
      },
      _opts || {}
    );
    let pdDocument = PdfLoader.load(obj.contentStream);
    let pdfTextStripper = new PDFTextStripper();
    let result = null;
    try {
      if (opts.fromPage) {
        pdfTextStripper.setStartPage(opts.fromPage);
      }
      if (opts.toPage) {
        pdfTextStripper.setEndPage(opts.toPage);
      }
      result = pdfTextStripper.getText(pdDocument);
    } catch (e) {
      pdDocument.close();
      throw e;
    } finally {
      pdDocument.close();
    }
    return result;
  }

  // Return the module functions
  return {
    loadFromObject: loadFromObject,
    mergeDocumentsFromQuery: mergeDocumentsFromQuery,
    extractText: extractText
  };
}

// Initialize $$ when the module is required without parameters
let moduleExports = moduleFunction();

// Export the module function and its methods
module.exports = Object.assign(moduleFunction, moduleExports);
